/*
********************************************************************************
*	
*   Copyright (c) 2017 by Industrial Control Communications, Inc.
*
*   This software is copyrighted by and is the sole property of
*   Industrial Control Communications, Inc.  Any unauthorized use,
*   duplication, transmission, distribution, or disclosure of this
*   software is expressly forbidden.
*
*   This Copyright notice may not be removed or modified without prior
*   written consent of Industrial Control Communications, Inc.
*
*   ICC, Inc.
*   230 Horizon Dr                      	USA 608.831.1255
*   Suite 100                               http://www.iccdesigns.com
*   Verona, WI 53593                     	support@iccdesigns.com
*
********************************************************************************
*
*          Project: SpiLinuxExample
*        File Name: SpiCommand.c
*    Original Date: 08.24.2017
*           Author: Josh Schulze
*
*      Description: This file contains functions used to send SPI commands under Linux.
*
*      				This program uses two types of commands:
*      				Managed Commands - These commands are used to simply read/write data
*      								   and are performed by sending various ICC SPI commands
*      								   while monitoring the response to properly progress through
*      								   the SPI states.
*      				Single Commands -  These low level commands are used to send a single ICC SPI
*      								   command to the device. This allows a user to step through
*      								   all SPI states and see the responses for each SPI command.
*
*
*   Edit Date           Edit Description
*   ===============     ========================================================
*   
*
********************************************************************************
*/

/*
********************************************************************************
*                                   INCLUDES
********************************************************************************
*/

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "cpu.h"
#include "UserIO.h"
#include "SpiCommand.h"
#include "IccSpiProtocol.h"

/*
********************************************************************************
*                                 LOCAL MACROS
********************************************************************************
*/

#define NUM_RETRIES				5		// the number of times to retry an ICC SPI command when sending a managed command

/*
********************************************************************************
*                              PRIVATE PROTOTYPES
********************************************************************************
*/

static void SendManagedCommand (int fd_spi, struct spi_ioc_transfer *p_spi_msg, SpiCommand *p_spi_cmd,
								SpiStatusReg *p_status_reg, uint32_t *p_msg_value);

static void SendSingleCommand (int fd_spi, struct spi_ioc_transfer *p_spi_msg, SpiCommand *p_spi_cmd,
								SpiStatusReg *p_status_reg, uint32_t *p_msg_value);

/*
********************************************************************************
*                               PUBLIC FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : OpenSpiConnection
* Description : Opens a  handle to the SPI device and sets the required settings.
* Arguments   : SpiConfig*	: the SPI configuration settings
* Returns     : The handle to the SPI device,
* 				otherwise an error code if open failed.
* Comments    :
********************************************************************************
*/
int OpenSpiConnection (SpiConfig *p_spi_cfg)
{
	int error_code = 0;
	int fd_spi = 0;


	// Open the SPI device
	fd_spi = open(SPI_DEVICE, O_RDWR);
	if(fd_spi < 0)
	{
		error_code = -1;
		printf("Failed to open the SPI device.\n");
	}

	// Set SPI mode
	if(error_code == 0)
	{
		error_code = ioctl(fd_spi, SPI_IOC_WR_MODE, &p_spi_cfg->mode);
		if(error_code == -1)
		{
			printf("Failed to set the mode.\n");
		}
	}

	// Set LSB first
	if(error_code == 0)
	{
		error_code = ioctl(fd_spi, SPI_IOC_WR_LSB_FIRST, &p_spi_cfg->lsb_first);
		if(error_code == -1)
		{
			printf("Failed to set the mode.\n");
		}
	}

	// Set bits per word
	if(error_code == 0)
	{
		error_code = ioctl(fd_spi, SPI_IOC_WR_BITS_PER_WORD, &p_spi_cfg->bits_per_word);
		if(error_code == -1)
		{
			printf("Failed to set the mode.\n");
		}
	}

	// Set the SPI max speed
	if(error_code == 0)
	{
		error_code = ioctl(fd_spi, SPI_IOC_WR_MAX_SPEED_HZ, &p_spi_cfg->speed_hz);
		if(error_code == -1)
		{
			printf("Failed to set the max speed.\n");
		}
	}

	return fd_spi;
}

/*
********************************************************************************
* Function    : SendSpiMessage
* Description : Transmit the SPI message and receive response.
* Arguments   : int							: the file descriptor for the SPI device
* 				struct spi_ioc_transfer* 	: the SPI IOC transfer settings
* Returns     : int	: the number of bytes sent,
* 					  otherwise an error code
* Comments    : Under Linux, the read() and write() file commands cannot be
* 				used because they only perform half-duplex transfers.
* 				Therefore, ioctl() is used to perform a full-duplex transfer.
* 				The spi_ioc_transfer structure defines the SPI transfer settings
* 				for this transaction.
********************************************************************************
*/
int SendSpiMessage (int fd_spi, struct spi_ioc_transfer *p_spi_msg)
{
	int bytes_sent = 0;


	// Send the message
	bytes_sent = ioctl(fd_spi, SPI_IOC_MESSAGE(1), p_spi_msg);

	if(bytes_sent < 1)
	{
		printf("Failed to send the message.\n");
		printf("ERROR: %s\n", strerror(errno));
	}

	return bytes_sent;
}

/*
********************************************************************************
* Function    : SpiEncodeMessage
* Description : Encodes the ICC SPI command into the transmission buffer.
* Arguments   : SpiCommand*	: the command to send
* 				uint8_t*	: the transmission buffer
* Returns     : None
* Comments    :
********************************************************************************
*/
void SpiEncodeMessage (SpiCommand *p_spi_cmd, uint8_t *buf)
{
	switch(p_spi_cmd->command)
	{
		case SPI_CMD_SET_ADDR:
			BuildIccSpiSetAddress(buf, p_spi_cmd->address);
			break;

		case SPI_CMD_READ_BYTE:
			BuildIccSpiRead(buf, 1);
			break;

		case SPI_CMD_READ_SHORT:
			BuildIccSpiRead(buf, 2);
			break;

		case SPI_CMD_READ_LONG:
			BuildIccSpiRead(buf, 4);
			break;

		case SPI_CMD_WRITE_BYTE:
			BuildIccSpiWrite(buf, 1, p_spi_cmd->value);
			break;

		case SPI_CMD_WRITE_SHORT:
			BuildIccSpiWrite(buf, 2, p_spi_cmd->value);
			break;

		case SPI_CMD_WRITE_LONG:
			BuildIccSpiWrite(buf, 4, p_spi_cmd->value);
			break;

		// SPI_CMD_GET_STATUS
		default:
			BuildIccSpiGetStatus(buf);
			break;
	}
}

/*
********************************************************************************
* Function    : SpiDecodeResponse
* Description : Decodes the received ICC SPI message.
* Arguments   : uint8_t*		: the receive buffer
* 				SpiStatusReg*	: Output - the value of the status response
* 				uint32_t*		: Output - the value of the data in the response
* Returns     : None
* Comments    :
********************************************************************************
*/
void SpiDecodeResponse (uint8_t *buf, SpiStatusReg *status_reg, uint32_t *msg_value)
{
	GetIccSpiResponseData(buf, status_reg, msg_value);
}

/*
********************************************************************************
* Function    : SendCommand
* Description : Sends a managed or single SPI command.
* Arguments   : int							: open handle to the SPI device
* 				struct spi_ioc_transfer* 	: the SPI IOC transfer settings
* 				SpiCommand*					: the command to send
* Returns     : None
* Comments    : This function prompts the user for confirmation before sending the command
********************************************************************************
*/
void SendCommand (int fd_spi, struct spi_ioc_transfer *p_spi_msg, SpiCommand *p_spi_cmd)
{
	bool run_cmd;
	SpiStatusReg 	status_reg;
	uint32_t		msg_value;


	// Wait for user input
	run_cmd = CanRunSpiCommand(p_spi_cmd);

	if (run_cmd)
	{
		if (p_spi_cmd->managed_cmd)
		{
			SendManagedCommand(fd_spi, p_spi_msg, p_spi_cmd, &status_reg, &msg_value);
		}
		else
		{
			SendSingleCommand(fd_spi, p_spi_msg, p_spi_cmd, &status_reg, &msg_value);

			// Display the response results to the user
			PrintResults(p_spi_cmd, &status_reg, msg_value);
		}
	}
}

/*
********************************************************************************
*                               PRIVATE FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : SendManagedCommand
* Description : Sends a sequence of SPI commands and checks the responses
* 				in order to complete a high-level managed command.
* Arguments   : int							: open handle to the SPI device
* 				struct spi_ioc_transfer* 	: the SPI IOC transfer settings
* 				SpiCommand*					: the command to send
* 				SpiStatusReg*				: Output - the value of the status response
* 				uint32_t*					: Output - the value of the data in the response
* Returns     : None
* Comments    :
********************************************************************************
*/
static void SendManagedCommand (int fd_spi, struct spi_ioc_transfer *p_spi_msg, SpiCommand *p_spi_cmd,
								SpiStatusReg *p_status_reg, uint32_t *p_msg_value)
{
	SpiCommand 		temp_cmd;
	int				retry_count;
	__useconds_t	t_deselect_us;


	// the PicoPort requires a TDeselect time of at least 1 clock period and at least 150us
	t_deselect_us = 1000000 / p_spi_msg->speed_hz;

	if (t_deselect_us < SPI_T_DESELECT_US)
		t_deselect_us = SPI_T_DESELECT_US;

	printf("Working...\n");

	// first, we must send the Set Address command
	bzero(&temp_cmd, sizeof(SpiCommand));
	temp_cmd.command = SPI_CMD_SET_ADDR;
	temp_cmd.address = p_spi_cmd->address;

	// loop until we receive an ACK or we reach our retry limit
	retry_count = 0;
	do
	{
		// make sure the chip select is deasserted for at least the minimum allowed
		usleep(t_deselect_us);
		SendSingleCommand(fd_spi, p_spi_msg, &temp_cmd, p_status_reg, p_msg_value);
		retry_count++;
	} while (p_status_reg->ack == 0 && retry_count <= NUM_RETRIES);

	// make sure the previous command succeeded
	if (retry_count <= NUM_RETRIES)
	{
		// next, send the Get Status command to get the current state
		bzero(&temp_cmd, sizeof(SpiCommand));
		temp_cmd.command = SPI_CMD_GET_STATUS;

		// loop until the device is in the ready state or we reach our retry limit
		retry_count = 0;
		do
		{
			// make sure the chip select is deasserted for at least the minimum allowed
			usleep(t_deselect_us);
			SendSingleCommand(fd_spi, p_spi_msg, &temp_cmd, p_status_reg, p_msg_value);
			retry_count++;
		} while (p_status_reg->state != SPI_STATE_READY && retry_count <= NUM_RETRIES);
	}

	// make sure the previous command succeeded
	if (retry_count <= NUM_RETRIES)
	{
		// next, send the read or write command
		bzero(&temp_cmd, sizeof(SpiCommand));
		temp_cmd.command = p_spi_cmd->command;
		temp_cmd.value = p_spi_cmd->value;

		// loop until we receive an ACK or we reach our retry limit
		retry_count = 0;
		do
		{
			// make sure the chip select is deasserted for at least the minimum allowed
			usleep(t_deselect_us);
			SendSingleCommand(fd_spi, p_spi_msg, &temp_cmd, p_status_reg, p_msg_value);
			retry_count++;
		} while (p_status_reg->ack == 0 && retry_count <= NUM_RETRIES);
	}

	// make sure the previous command succeeded
	if (retry_count <= NUM_RETRIES)
	{
		// finally, send the Get Status command to get the current state
		bzero(&temp_cmd, sizeof(SpiCommand));
		temp_cmd.command = SPI_CMD_GET_STATUS;

		// loop until the device is in the operation complete state or we reach our retry limit
		retry_count = 0;
		do
		{
			// make sure the chip select is deasserted for at least the minimum allowed
			usleep(t_deselect_us);
			SendSingleCommand(fd_spi, p_spi_msg, &temp_cmd, p_status_reg, p_msg_value);
			retry_count++;
		} while (p_status_reg->state != SPI_STATE_OPERATION_COMPLETE && retry_count <= NUM_RETRIES);
	}

	// check if we succeeded
	if (retry_count <= NUM_RETRIES)
	{
		// Display the value read/written to the user
		printf("Success\n");
		PrintData(p_spi_cmd, p_status_reg, *p_msg_value);
	}
	else
	{
		// Display the current status and data
		printf("Failed\n");
		PrintStatus(p_status_reg);
		PrintData(p_spi_cmd, p_status_reg, *p_msg_value);
	}
}

/*
********************************************************************************
* Function    : SendSingleCommand
* Description : Sends a single, low level ICC SPI command
* Arguments   : int							: open handle to the SPI device
* 				struct spi_ioc_transfer* 	: the SPI IOC transfer settings
* 				SpiCommand*					: the command to send
* 				SpiStatusReg*				: Output - the value of the status response
* 				uint32_t*					: Output - the value of the data in the response
* Returns     : None
* Comments    :
********************************************************************************
*/
static void SendSingleCommand (int fd_spi, struct spi_ioc_transfer *p_spi_msg, SpiCommand *p_spi_cmd,
								SpiStatusReg *p_status_reg, uint32_t *p_msg_value)
{
	uint8_t			*tx_buf;
	uint8_t			*rx_buf;


	// get the buffer pointers from the spi msg struct
	tx_buf = (uint8_t*)((uintptr_t)p_spi_msg->tx_buf);
	rx_buf = (uint8_t*)((uintptr_t)p_spi_msg->rx_buf);

	// Encode the command
	SpiEncodeMessage(p_spi_cmd, tx_buf);

	// Send the message
	SendSpiMessage(fd_spi, p_spi_msg);

	// Decode the response
	SpiDecodeResponse(rx_buf, p_status_reg, p_msg_value);
}
