/*
********************************************************************************
*	
*   Copyright (c) 2017 by Industrial Control Communications, Inc.
*
*   This software is copyrighted by and is the sole property of
*   Industrial Control Communications, Inc.  Any unauthorized use,
*   duplication, transmission, distribution, or disclosure of this
*   software is expressly forbidden.
*
*   This Copyright notice may not be removed or modified without prior
*   written consent of Industrial Control Communications, Inc.
*
*   ICC, Inc.
*   230 Horizon Dr                      	USA 608.831.1255
*   Suite 100                               http://www.iccdesigns.com
*   Verona, WI 53593                     	support@iccdesigns.com
*
********************************************************************************
*
*          Project: SpiLinuxExample
*        File Name: UserIO.c
*    Original Date: 08.24.2017
*           Author: Josh Schulze
*
*      Description: Contains functions for interacting with a user via command line
*      				on Linux in order to send SPI messages to a device.
*
*
*   Edit Date           Edit Description
*   ===============     ========================================================
*   
*
********************************************************************************
*/

/*
********************************************************************************
*                                   INCLUDES
********************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "SpiLinuxExample.h"
#include "IccSpiProtocol.h"
#include "SpiCommand.h"
#include "UserIO.h"

/*
********************************************************************************
*                                LOCAL VARIABLES
********************************************************************************
*/

// Array of SPI State names
const char *mSpiStateNames[] =
{
	"Reset",
	"Busy",
	"Ready",
	"Operation Complete",
};

/*
********************************************************************************
*                              PRIVATE PROTOTYPES
********************************************************************************
*/

static void ParseSpiCommandLine (char *cmd_line, SpiCommand *spi_cmd);

static int GetCommandOption (char *cmd_str, char *opt_str);

static uint8_t GetSpiCommandFromString (char *cmd_str);

static char* GetSpiCommandName (SpiCommands command);

static char* GetSpiErrorString (uint8_t error_code);

static bool IsWhiteSpace (char c);

/*
********************************************************************************
*                               PUBLIC FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : PrintSpiPortSettings
* Description : Prints the SPI port configuration.
* Arguments   : SpiConfig*	: the SPI configuration
* Returns     : None
* Comments    :
********************************************************************************
*/
void PrintSpiPortSettings (SpiConfig *p_spi_cfg)
{
	float speed_khz = ((float)p_spi_cfg->speed_hz / (float)1000);
	float speed_mhz = ((float)p_spi_cfg->speed_hz / (float)1000000);

	printf("\n");
	printf("SPI Port Settings:\n");
	printf("  Mode: %d\n", p_spi_cfg->mode);
	printf("  Bit Justification: %s\n", p_spi_cfg->lsb_first ? "LSB-first" : "MSB-first");
	printf("  Bits per Word: %d\n", p_spi_cfg->bits_per_word);
	printf("  Clock Speed: %d Hz (%0.3f KHz) (%0.6f MHz)\n", p_spi_cfg->speed_hz, speed_khz, speed_mhz);
}

/*
********************************************************************************
* Function    : PrintHelp
* Description : Prints usage information and application command help.
* Arguments   : bool	: true to print usage information
* 						  false to print application command help
* Returns     : None
* Comments    :
********************************************************************************
*/
void PrintHelp (bool show_usage)
{
	if (show_usage)
	{
		printf("Usage: SpiLinuxExample [OPTION]...\n");
		printf("Send ICC SPI commands to a connected PicoPort.\n");
		printf("With no OPTION, same as -s 1500 -m 0.\n");
		printf("\n");
		printf("  -s [SPEED]\t\t\t Set the SPI speed in Hz\n");
		printf("  -m [MODE]\t\t\t Set the SPI mode\n");
		printf("  \t\t\t\t 0 - Mode 0\n");
		printf("  \t\t\t\t 1 - Mode 1\n");
		printf("  \t\t\t\t 2 - Mode 2\n");
		printf("  \t\t\t\t 3 - Mode 3\n");
	}
	else
	{
		printf("\n");
		printf("User SPI Commands\n");
		printf("-------------------------\n");
		printf("  Managed commands to send the proper low level SPI commands and\n");
		printf("  progress through ICC SPI states to read or write data.\n");
		printf("\n");
		printf("  %-5s [B/S/L] [Addr*] \t Read Byte, Short, or Long\n", SPI_SCMD_MNG_READ);
		printf("  %-5s [B/S/L] [Addr*] [Value*] Write Byte, Short, or Long\n", SPI_SCMD_MNG_WRITE);
		printf("\n");
		printf("Low Level SPI Commands\n");
		printf("-------------------------\n");
		printf("  Send individual ICC SPI commands to the device.\n");
		printf("\n");
		printf("  %s\t\t\t\t Get Status\n", SPI_SCMD_GET_STATUS);
		printf("\n");
		printf("  %s [Addr*]\t\t\t Set Address\n", SPI_SCMD_SET_ADDR);
		printf("\n");
		printf("  %s \t\t\t\t Read Byte\n", SPI_SCMD_READ_BYTE);
		printf("  %s \t\t\t\t Read Short\n", SPI_SCMD_READ_SHORT);
		printf("  %s \t\t\t\t Read Long\n", SPI_SCMD_READ_LONG);
		printf("\n");
		printf("  %s [Value*]\t\t\t Write Byte\n", SPI_SCMD_WRITE_BYTE);
		printf("  %s [Value*]\t\t\t Write Short\n", SPI_SCMD_WRITE_SHORT);
		printf("  %s [Value*]\t\t\t Write Long\n", SPI_SCMD_WRITE_LONG);
		printf("\n");
		printf("\n");
		printf("* Numbers may be entered in decimal or\n");
		printf("  optionally in hexadecimal using the '0x' prefix\n");
	}
}

/*
********************************************************************************
* Function    : PrintResults
* Description : Prints the status and data results of an SPI command.
* Arguments   : SpiCommand*		: the SPI command that was sent
* 				SpiStatusReg*	: the status register
* 				uint32_t		: the data value
* Returns     : None
* Comments    :
********************************************************************************
*/
void PrintResults (SpiCommand *p_spi_cmd, SpiStatusReg *p_status_reg, uint32_t spi_val)
{
	// Print Status
	PrintStatus(p_status_reg);

	// Print Data
	PrintData(p_spi_cmd, p_status_reg, spi_val);
}

/*
********************************************************************************
* Function    : PrintStatus
* Description : Prints the ICC SPI status register.
* Arguments   : SpiStatusReg*	: the status register
* Returns     : None
* Comments    :
********************************************************************************
*/
void PrintStatus (SpiStatusReg *p_status_reg)
{
	// Print Status
	printf("Status Register:\n");
	printf("\t state = %s\n", mSpiStateNames[p_status_reg->state]);
	printf("\t ACK = %d\n", p_status_reg->ack);
	printf("\t ERR = %d\n", p_status_reg->err);
}

/*
********************************************************************************
* Function    : PrintData
* Description : Prints the ICC SPI data or error value.
* Arguments   : SpiCommand*		: the SPI command that was sent
* 				SpiStatusReg*	: the status register
* 				uint32_t		: the data value
* Returns     : None
* Comments    : This function detects whether the response was an error in order
* 				to print the error code. It also uses the SPI command to determine
* 				the size of the value to print for read/write operations.
********************************************************************************
*/
void PrintData (SpiCommand *p_spi_cmd, SpiStatusReg *p_status_reg, uint32_t spi_val)
{
	// Print Data
	if(p_status_reg->err)
	{
		printf("Error: %s (0x%02X)\n", GetSpiErrorString((uint8_t)spi_val), (uint8_t)spi_val);
	}
	else
	{
		// print the correct sized value for managed command read/writes
		if (p_spi_cmd->managed_cmd)
		{
			switch (p_spi_cmd->command)
			{
				case SPI_CMD_READ_BYTE:
				case SPI_CMD_WRITE_BYTE:
					printf("Value: %d (0x%02X)\n", spi_val, spi_val);
					break;

				case SPI_CMD_READ_SHORT:
				case SPI_CMD_WRITE_SHORT:
					printf("Value: %d (0x%04X)\n", spi_val, spi_val);
					break;

				default:
					printf("Value: %d (0x%08X)\n", spi_val, spi_val);
					break;
			}
		}
		else
		{
			printf("Value: %d (0x%08X)\n", spi_val, spi_val);
		}
	}
}

/*
********************************************************************************
* Function    : GetSpiCommandFromUser
* Description : Reads a user command from stdin.
* Arguments   : SpiCommand*	: Output - the normalized SPI command parsed from the user's commmand
* 				bool*		: Output - whether the user's command was valid or not
* Returns     : bool	: true to continue running the program
* 						  false to quit the program
* Comments    :
********************************************************************************
*/
bool GetSpiCommandFromUser (SpiCommand *p_spi_cmd, bool *p_cmd_valid)
{
	bool ret = true;
	bool get_cmd = true;
	int c;
	int idx = 0;
	char cmd_str[51];


	// Clear the command struct
	bzero(p_spi_cmd, sizeof(SpiCommand));

	// Read user input
	while(get_cmd)
	{
		idx = 0;

		printf("\n");
		printf("Type 'h' for help or 'q' to quit.\n");
		printf("Enter a command: ");
		do
		{
			c = toupper(getchar());
			cmd_str[idx++] = (char)c;
		}while(c != '\n' && idx < 50);

		cmd_str[idx] = '\0';	// null terminate

		switch(cmd_str[0])
		{
			case 'H':
				// Print help for available commands
				PrintHelp(false);
				get_cmd = true;
				break;

			case 'Q':
				ret = false;
				get_cmd = false;
				break;

			default:
				get_cmd = false;
				break;
		}
	}

	// If a command was entered, try to parse it
	if(ret)
	{
		ParseSpiCommandLine(cmd_str, p_spi_cmd);

		*p_cmd_valid = (p_spi_cmd->command != 0);

		if (!*p_cmd_valid)
			printf("Unknown command\n");
	}

	return ret;
}

/*
********************************************************************************
* Function    : CanRunSpiCommand
* Description : Prompts the user for confirmation on whether a command
* 				should be sent to the slave device.
* Arguments   : SpiCommand*	: the SPI command to send
* Returns     : bool	: whether or not the command should be executed
* Comments    :
********************************************************************************
*/
bool CanRunSpiCommand (SpiCommand *p_spi_cmd)
{
	bool run_cmd = true;
	int usr_in;


	if (p_spi_cmd->managed_cmd)
	{
		switch (p_spi_cmd->command)
		{
			case SPI_CMD_READ_BYTE:
				printf("Read byte at address %d (0x%04X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address);
				break;

			case SPI_CMD_READ_SHORT:
				printf("Read short at address %d (0x%04X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address);
				break;

			case SPI_CMD_READ_LONG:
				printf("Read long at address %d (0x%04X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address);
				break;

			case SPI_CMD_WRITE_BYTE:
				printf("Write byte at address %d (0x%04X) to %d (0x%02X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address, p_spi_cmd->value, p_spi_cmd->value);
				break;

			case SPI_CMD_WRITE_SHORT:
				printf("Write short at address %d (0x%04X) to %d (0x%04X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address, p_spi_cmd->value, p_spi_cmd->value);
				break;

			case SPI_CMD_WRITE_LONG:
				printf("Write long at address %d (0x%04X) to %d (0x%08X) (Y/n)? ", p_spi_cmd->address, p_spi_cmd->address, p_spi_cmd->value, p_spi_cmd->value);
				break;

			default:
				printf("Send %s command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command));
				break;
		}
	}
	else
	{
		switch (p_spi_cmd->command)
		{
			case SPI_CMD_SET_ADDR:
				printf("Send %s %d (0x%04X) command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command), p_spi_cmd->address, p_spi_cmd->address);
				break;

			case SPI_CMD_WRITE_BYTE:
				printf("Send %s %d (0x%02X) command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command), p_spi_cmd->value, p_spi_cmd->value);
				break;

			case SPI_CMD_WRITE_SHORT:
				printf("Send %s %d (0x%04X) command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command), p_spi_cmd->value, p_spi_cmd->value);
				break;

			case SPI_CMD_WRITE_LONG:
				printf("Send %s %d (0x%08X) command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command), p_spi_cmd->value, p_spi_cmd->value);
				break;

			default:
				printf("Send %s command (Y/n)? ", GetSpiCommandName(p_spi_cmd->command));
				break;
		}
	}

	do
	{
		usr_in = getchar();

		if ('n' == (char) usr_in || 'N' == (char) usr_in)
			run_cmd = false;
	} while (usr_in != '\n');

	return run_cmd;
}

/*
********************************************************************************
*                               PRIVATE FUNCTIONS
********************************************************************************
*/

/*
********************************************************************************
* Function    : ParseSpiCommandLine
* Description : Parses the user's command into a normalized SPI command.
* Arguments   : char*		: the user's command string
* 				SpiCommand*	: Output - the normalized SPI command
* Returns     : None
* Comments    :
********************************************************************************
*/
static void ParseSpiCommandLine (char *cmd_line, SpiCommand *p_spi_cmd)
{
	char opt_str[80];
	int idx = 0;
	uint8_t temp_uint8;
	uint16_t temp_uint16;


	// Get the command from the command string
	idx += GetCommandOption(&cmd_line[idx], opt_str);

	// default command to GS
	if(strlen(opt_str) == 0)
	{
		p_spi_cmd->command = SPI_CMD_GET_STATUS;
	}
	else if(strcasecmp(opt_str, SPI_SCMD_MNG_READ) == 0)
	{
		p_spi_cmd->managed_cmd = true;

		// Get the read type from the command string
		idx += GetCommandOption(&cmd_line[idx], opt_str);

		if (strcasecmp(opt_str, "B") == 0)
		{
			p_spi_cmd->command = SPI_CMD_READ_BYTE;
		}
		else if (strcasecmp(opt_str, "S") == 0)
		{
			p_spi_cmd->command = SPI_CMD_READ_SHORT;
		}
		else if (strcasecmp(opt_str, "L") == 0)
		{
			p_spi_cmd->command = SPI_CMD_READ_LONG;
		}
		else
		{
			// invalid option
			p_spi_cmd->command = 0;
		}
	}
	else if(strcasecmp(opt_str, SPI_SCMD_MNG_WRITE) == 0)
	{
		p_spi_cmd->managed_cmd = true;

		// Get the write type from the command string
		idx += GetCommandOption(&cmd_line[idx], opt_str);

		if (strcasecmp(opt_str, "B") == 0)
		{
			p_spi_cmd->command = SPI_CMD_WRITE_BYTE;
		}
		else if (strcasecmp(opt_str, "S") == 0)
		{
			p_spi_cmd->command = SPI_CMD_WRITE_SHORT;
		}
		else if (strcasecmp(opt_str, "L") == 0)
		{
			p_spi_cmd->command = SPI_CMD_WRITE_LONG;
		}
		else
		{
			// invalid option
			p_spi_cmd->command = 0;
		}
	}
	else
	{
		p_spi_cmd->command = GetSpiCommandFromString(opt_str);
	}

	// Read Address
	if (p_spi_cmd->managed_cmd || p_spi_cmd->command == SPI_CMD_SET_ADDR)
	{
		// Parse the address
		idx += GetCommandOption(&cmd_line[idx], opt_str);

		sscanf(opt_str, "%hi", &p_spi_cmd->address);
	}

	// Read Value
	switch (p_spi_cmd->command)
	{
		case SPI_CMD_WRITE_BYTE:
		case SPI_CMD_WRITE_SHORT:
		case SPI_CMD_WRITE_LONG:
			// Parse the value
			idx += GetCommandOption(&cmd_line[idx], opt_str);

			// Check if there was an option
			if (strcmp(opt_str, ""))
			{
				switch (p_spi_cmd->command)
				{
					case SPI_CMD_WRITE_BYTE:
						sscanf(opt_str, "%hhi", &temp_uint8);
						p_spi_cmd->value = temp_uint8;
						break;

					case SPI_CMD_WRITE_SHORT:
						sscanf(opt_str, "%hi", &temp_uint16);
						p_spi_cmd->value = temp_uint16;
						break;

					default: 			// SPI_CMD_WRITE_LONG
						sscanf(opt_str, "%i", &p_spi_cmd->value);
						break;
				}
			}
			else
			{
				p_spi_cmd->value = 0;
			}
			break;

		default:
			//Do nothing
			break;
	}
}

/*
********************************************************************************
* Function    : GetCommandOption
* Description : Gets the next command line option separated by whitespace.
* Arguments   : char*	: the remaining command line string
* 				char*	: Output - the parsed option
* Returns     : int	: the number of bytes parsed from the command line string
* Comments    :
********************************************************************************
*/
static int GetCommandOption (char *cmd_str, char *opt_str)
{
	int bytes_read 	= 0;
	int opt_len		= 0;

	// ignore white space
	while (IsWhiteSpace(cmd_str[bytes_read]))
	{
		bytes_read++;
	}

	// Parse the string, stopping at a new line
	while(!IsWhiteSpace(cmd_str[bytes_read]) && cmd_str[bytes_read] != '\n')
	{
		opt_str[opt_len++] = cmd_str[bytes_read++];
	}

	opt_str[opt_len] = '\0';	// null terminate

	return bytes_read;
}

/*
********************************************************************************
* Function    : GetSpiCommandFromString
* Description : Converts a user command string into an enumerated SPI command.
* Arguments   : char*	: the user command string
* Returns     : uint8_t	: the ICC SPI command
* Comments    :
********************************************************************************
*/
static uint8_t GetSpiCommandFromString (char *cmd_str)
{
	uint8_t spi_cmd;


	if(strcasecmp(cmd_str, SPI_SCMD_GET_STATUS) == 0)
	{
		spi_cmd = SPI_CMD_GET_STATUS;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_SET_ADDR) == 0)
	{
		spi_cmd = SPI_CMD_SET_ADDR;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_READ_BYTE) == 0)
	{
		spi_cmd = SPI_CMD_READ_BYTE;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_READ_SHORT) == 0)
	{
		spi_cmd = SPI_CMD_READ_SHORT;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_READ_LONG) == 0)
	{
		spi_cmd = SPI_CMD_READ_LONG;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_WRITE_BYTE) == 0)
	{
		spi_cmd = SPI_CMD_WRITE_BYTE;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_WRITE_SHORT) == 0)
	{
		spi_cmd = SPI_CMD_WRITE_SHORT;
	}
	else if(strcasecmp(cmd_str, SPI_SCMD_WRITE_LONG) == 0)
	{
		spi_cmd = SPI_CMD_WRITE_LONG;
	}
	else
	{
		// invalid command
		spi_cmd = 0;
	}

	return spi_cmd;
}

/*
********************************************************************************
* Function    : GetSpiCommandName
* Description : Get the human-readable name of an ICC SPI command.
* Arguments   : SpiCommands	: the ICC SPI command
* Returns     : char*	: a string containing the name of the command
* Comments    :
********************************************************************************
*/
static char* GetSpiCommandName (SpiCommands command)
{
	char *cmd_name = NULL;

	switch((int)command)
	{
		case SPI_CMD_GET_STATUS:
			cmd_name = "Get Status";
			break;

		case SPI_CMD_SET_ADDR:
			cmd_name = "Set Address";
			break;

		case SPI_CMD_READ_BYTE:
			cmd_name = "Read Byte";
			break;

		case SPI_CMD_READ_SHORT:
			cmd_name = "Read Short";
			break;

		case SPI_CMD_READ_LONG:
			cmd_name = "Read Long";
			break;

		case SPI_CMD_WRITE_BYTE:
			cmd_name = "Write Byte";
			break;

		case SPI_CMD_WRITE_SHORT:
			cmd_name = "Write Short";
			break;

		case SPI_CMD_WRITE_LONG:
			cmd_name = "Write Long";
			break;

		default:
			cmd_name = "Unknown";
			break;
	}

	return cmd_name;
}

/*
********************************************************************************
* Function    : GetSpiErrorString
* Description : Gets the human-readable error string from the error code.
* Arguments   : uint8_t	: the error code
* Returns     : char*	: a string containing the error
* Comments    :
********************************************************************************
*/
static char* GetSpiErrorString (uint8_t error_code)
{
	char *err_str = NULL;

	switch(error_code)
	{
		case SPI_ERR_INVALID_ADDR:
			err_str = "Invalid Address";
			break;

		case SPI_ERR_DATA_ERROR:
			err_str = "Data Error";
			break;

		case SPI_ERR_WR_TO_RO:
			err_str = "Write to Read-Only";
			break;

		case SPI_ERR_RD_FROM_WO:
			err_str = "Read from Write-Only";
			break;

		case SPI_ERR_INVALID_FUNC:
			err_str = "Invalid Function";
			break;

		case SPI_ERR_INVALID_PKT:
			err_str = "Invalid Packet";
			break;

		default:
			err_str = "Unknown Error";
			break;
	}

	return err_str;
}

/*
********************************************************************************
* Function    : IsWhiteSpace
* Description : Determines if a character is whitespace.
* Arguments   : char	: the character to check
* Returns     : bool	: true if the character is whitespace
* Comments    :
********************************************************************************
*/
static bool IsWhiteSpace (char c)
{
	return (c == ' ' || c == '\t');
}
